include <Chamfers-for-OpenSCAD/Chamfer.scad>;

module brickpattern(length = 80, 
                    height = 18, 
                    brickwidth = 7.5, 
                    brickheight = 2.5, 
                    brickthickness = 3.5, 
                    mortar = 0.5, 
                    shift = 0, 
                    rand = 0, 
                    chamfer = 0.5, 
                    debug = false) {
    /* brickpattern - makes brick wall patterns
    Usage:
        use <path/to/brickpattern.scad>
        ...
        brickpattern(
            length = 80         // The length of the wall to be "bricked"
            height = 18         // The height of the wall to be "bricked", height should be
                                //  an even multiple of the sum of brickheight and mortar
            brickwidth = 7.5    // individual brick width
            brickheight = 2.5   // individual brick height
            brickthickness = 3.5// individual brick thickness
            mortar = 0.5        // width of mortar lines between bricks
            shift = 0           // start row has half-bricks on ends
            rand = 0            // rotate each brick a random angle between 
                                //  -rand and rand, in degrees
            chamfer = 0.5       // Setting to a number greater than 0 chamfers the exposed 
                                //  edges of the bricks.  Beneficial when 3D printing at 
                                //  small scales - HO and smaller.
            debug = false       // If true, echos debug messages to the console pane.
        );
        
    The numbers given above are the defaults if you just do
    brickpattern();  define them to suit your scale and intended 
    brick style.
        
    This routine just draws the bricks; you need to add a
    cube of "mortar" in the dimensions of your wall into which
    to embed the pattern.
    To use in other programs, create your brick pattern in the 
    required dimensions and export as a .STL file.  Note that
    this will lose the color definition.
    */
    
    layers = height/(brickheight + mortar);             // The number of layers to cover 
                                                        //  the height of the wall
    if (debug) echo ("layers = ", layers);
    
    run = (brickwidth == length) ? 1 : floor(length/(brickwidth + mortar));
                                                        // The number of bricks to cover
                                                        //  the length of the wall                                                        
                                                        
    if (debug) echo ("run = ", run);
        
    difference = (brickwidth == length) ? 0 : length/(brickwidth + mortar) - run;
                                                        // The number of additional bricks 
                                                        //  needed to complete the wall
    
    if (debug) echo ("difference = ", difference);
    
    for (layer = [0:1:layers - 1]) {
		reg = shift ? !(layer % 2) : layer % 2;
		for (brick = [0:1:run-1]) {
			if (reg) {                                  //layer with half-bricks on each end
				if (brick == 0) {                       //first brick, half-brick
                    if (debug) echo("layer ", layer, " branch first brick, half brick");
					translate([0, 
                               0, 
							   layer * (brickheight + mortar)]) 
						color("darkred") 
                            rotate([0,
                                    rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([(brickwidth - mortar)/2, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [0, 1, 0, 0]], 
                                             chamfer);
				}
				else if (brick == run - 1) {            //last two bricks, a full one 
                                                        // and ending with a half one...
                    if (debug) echo ("layer ", layer, "branch last two bricks, one full, one half");
					translate([(brickwidth/2 + mortar/2) + (brick) * (brickwidth + mortar), 
							    0, 
								layer * (brickheight + mortar)]) 
						color("darkred") 
                            rotate([0,
                                    rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([(brickwidth - mortar)/2, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [1, 0, 0, 0]], 
                                             chamfer);
                    
					translate([(brickwidth/2 + mortar/2) + (brick - 1) * (brickwidth + mortar), 
							    0, 
								layer * (brickheight + mortar)])
						color("darkred") 
                            rotate([0,
                                    rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([brickwidth, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [1, 1, 0, 0]], 
                                             chamfer);
				}
				else {                                  //second and subsequent full bricks
                    if (debug) echo ("layer ", layer, "branch interior full bricks");
					translate([(brickwidth/2 + mortar/2) + (brick - 1) * (brickwidth + mortar), 
							   0, 
							   layer * (brickheight + mortar)]) 
						color("darkred") 
                            rotate([0,
                                    rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([brickwidth, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [1, 1, 0, 0]], 
                                             chamfer);
				}
			}
			else {                                      // layer that doesn't contain half-bricks
                if (debug) echo("layer ", layer, "branch full row, full bricks");
				translate([brick * (brickwidth + mortar),
							0, 
							layer * (brickheight + mortar)]) 
					color("darkred") 
                        rotate([0,
                                rands(-rand, rand, 1)[0], 0]) 
                            chamferCube([brickwidth, 
                                         brickthickness, 
                                         brickheight], 
                                        [[1, 0, 0, 1], 
                                         [0, 0, 0, 0], 
                                         [1, 1, 0, 0]], 
                                         chamfer);
			}
		}
        if (reg && run == 1) {
            if (debug) echo("layer = ", layer, "adding half brick to shifted row on single brick run");
            translate([(brickwidth/2 + mortar/2), 
                        0, 
                        layer * (brickheight + mortar)]) 
                color("darkred") 
                    rotate([0,
                            rands(-rand, rand, 1)[0], 0]) 
                        chamferCube([(brickwidth - mortar)/2, 
                                     brickthickness, 
                                     brickheight], 
                                    [[1, 0, 0, 1], 
                                     [0, 0, 0, 0], 
                                     [1, 0, 0, 0]], 
                                     chamfer);
        }
	}
    
    // Code to add remaining bricks at end of wall
    if (difference > 0) {
        if (debug) echo ("difference > 0, adding remaining bricks");
        remainingbw = difference * brickwidth;          // The length of the remaining brick
        if (debug) echo( "remaingbw = ", remainingbw, "bricklength/2 = ", brickwidth/2);
        finalBrickPos = run * (brickwidth + mortar) - mortar;   // The position to begin filling in
                                                                //  remaing bricks
        if (debug) echo("finalBrickPos = ", finalBrickPos);
        
        for (layer = [0:1:layers - 1]) {
            reg = shift ? !(layer % 2) : layer % 2;
            if (debug) echo("reg = ", reg);
            if (reg) {                                  // Layers with half bricks on each end
                if ((remainingbw + mortar) > brickwidth/2) {
                    if (debug) echo("layer ", layer, "adding bricks to half bricks where remaining is > half-brick length + mortar");
 					translate([finalBrickPos, 
							   0, 
							   layer * (brickheight + mortar)]) 
						color("darkred") 
                            rotate([0,
                                    rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([brickwidth/2, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [0, 1, 0, 0]], 
                                             chamfer);

                    if (remainingbw > brickwidth/2) {
                    if (debug) echo("layer ", layer, "adding bricks to half bricks where remaining is > half-brick length");
                        translate([finalBrickPos + (brickwidth/2 + mortar), 
								   0, 
								   layer * (brickheight + mortar)]) 
                            color("darkred") 
                                rotate([0,
                                        rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([(remainingbw - brickwidth/2), 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [1, 0, 0, 0]], 
                                             chamfer);
                    }
                }
                else if (remainingbw + mortar > brickwidth/2){
                    if (debug) echo("layer ", layer, "adding bricks where remaining is > half-brick length");
                        translate([finalBrickPos + mortar, 
								   0, 
								   layer * (brickheight + mortar)]) 
                            color("darkred") 
                                rotate([0,
                                        rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([(remainingbw - brickwidth/2), 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [1, 0, 0, 0]], 
                                             chamfer);
                }
                else if (remainingbw == brickwidth/2) {
                    if (debug) echo("layer ", layer, "adding bricks to half bricks where remaining is == half-brick length");
					translate([finalBrickPos, 
							   0, 
							   layer * (brickheight + mortar)]) 
						color("darkred")
                            rotate([0,
                                rands(-rand, rand, 1)[0], 0]) 
                                    chamferCube([brickwidth/2, 
                                                 brickthickness, 
                                                 brickheight], 
                                                [[1, 0, 0, 1], 
                                                 [0, 0, 0, 0], 
                                                 [0, 1, 0, 0]], 
                                                 chamfer);                   
                }
                else {
                    if (debug) echo("layer ", layer, "adding bricks to half bricks where remaining is < half-brick length");
 					translate([finalBrickPos, 
							   0, 
							   layer * (brickheight + mortar)]) 
						color("darkred") 
                            rotate([0,
                                rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([remainingbw + mortar, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [0, 0, 0, 0]], 
                                             chamfer);                    
                }
            }
            else {                                      // Layers without half bricks on each end
                if (remainingbw < (brickwidth + mortar)) {
                    if (debug) echo("layer ", layer, "adding bricks to rows with full bricks where remaining is < brick length + mortar");
                    translate([finalBrickPos + mortar, 
							   0, 
							   layer * (brickheight + mortar)]) 
						color("darkred") 
                            rotate([0,
                                    rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([remainingbw, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [1, 0, 0, 0]], 
                                             chamfer);
                }
                else if (remainingbw == (brickwidth + mortar)) {
                if (debug) echo("layer ", layer, "adding bricks to rows with full bricks where remaining is == brick length + mortar");
  					translate([finalBrickPos + mortar, 
							   0, 
							   layer * (brickheight + mortar)]) 
						color("darkred") 
                            rotate([0,
                                    rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([brickwidth, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [1, 1, 0, 0]], 
                                             chamfer);
                }
                else {
                if (debug) echo("layer ", layer, "adding bricks to rows with full bricks where remaining is > brick length + mortar");
   					translate([	finalBrickPos + mortar, 
								0, 
								layer * (brickheight + mortar)]) 
						color("darkred") 
                            rotate([0,rands(-rand, rand, 1)[0], 0]) 
                                chamferCube([brickwidth, 
                                             brickthickness, 
                                             brickheight], 
                                            [[1, 0, 0, 1], 
                                             [0, 0, 0, 0], 
                                             [1, 1, 0, 0]], 
                                             chamfer);
 
                    if (remainingbw > (brickwidth - mortar)){
                        translate([	finalBrickPos + mortar + brickwidth + mortar, 
								0, 
								layer * (brickheight + mortar)]) 
                            color("darkred") 
                                rotate([0,rands(-rand, rand, 1)[0], 0]) 
                                    chamferCube([remainingbw - brickwidth - mortar, 
                                                 brickthickness, 
                                                 brickheight], 
                                                [[1, 0, 0, 1], 
                                                 [0, 0, 0, 0], 
                                                 [1, 0, 0, 0]], 
                                                 chamfer);
                    }
                }
            }
        }
    }
}
